﻿Imports System
Imports System.Collections
Imports System.Threading

' Interfejs implementowany przez pule obiektów.
Public Interface IObjectPool(Of objectType)
    ' Liczba obiektów w puli.
    ReadOnly Property Count() As Integer

    ' Zwraca obiekt składowany w puli lub wartość null, jeśli pula jest pusta.
    Function GetObject() As objectType

    ' Zwraca obiekt składowany w puli. Jeśli pula jest pusta, czeka
    ' do momentu, w którym jakiś obiekt zostanie do niej zwrócony.
    Function WaitForObject() As objectType

    ' Zwraca obiekt do puli.
    Sub Release(ByVal o As objectType)
End Interface 'IObjectPool

' AbstractObjectPool jest abstrakcyjną klasą bazową dla klas implementujących
' interfejs IObjectPool. Klasa definiuje wspólną logikę dla wszystkich potomnych
' pul obiektów.
Public MustInherit Class AbstractObjectPool(Of objectType)
    Implements IObjectPool(Of objectType)

    ' Ponieważ klasa AbstractObjectPool nie została napisana z myślą o żadnej
    ' konkretnej klasie, nie zaimplementowano w niej mechanizmu tworzenia obiektów
    ' tej klasy (które mają być przedmiotem zarządzania w ramach puli). Zamiast tego
    ' odpowiedzialność za tworzenie obiektów jest delegowana do obiektu klasy
    ' implementującej interfejs nazwany ICreation. Kod interfejsu ICreation przedstawiono
    ' na końcu tego podrozdziału.
    Private creator As ICreation(Of objectType)

    ' creator - Obiektobiekt, do którego dana pula będzie delegowała zadanie tworzenia
    '           obiektów zarządzanych w ramach tej puli.
    Public Sub New(ByVal creator As ICreation(Of objectType))
        Me.creator = creator
    End Sub

    ' Obiekt odpowiedzialny za blokowanie dostępu i — tym samym — gwarantujący, że
    ' dostęp do struktury danych tej puli będzie miał jednocześnie tylko jeden wątek.
    Protected MustOverride ReadOnly Property SyncRoot() _
                                             As Object

    ' Zwraca wartość true, jeśli dany obiekt puli może tworzyć inne
    ' obiekty będące przedmiotem zarządzania w ramach tej puli.
    Protected MustOverride Function OkToCreate() As Boolean

    '
    ' Tworzy obiekt, który będzie zarządzany w ramach tej puli obiektów.
    '
    Protected Overridable Function createObject() As objectType
        Return creator.Create()
    End Function 'createObject

    ' Usuwa obiekt z tablicy reprezentującej pulę obiektów i zwraca go na wyjściu.
    ' Jeśli pula jest pusta, zwraca wartość Nothing.
    Protected MustOverride Function removeObject() As objectType

    ' Reprezentuje liczbę obiektów składowanych w puli.
    Public MustOverride ReadOnly Property Count() As Integer _
                              Implements IObjectPool(Of objectType).Count

    '
    ' Zwraca obiekt składowany w puli. Jeśli w danej chwili pula nie zawiera
    ' żadnych obiektów, funkcja tworzy nowy obiekt (chyba że stosowana w tej
    ' puli polityka tworzenia obiektów uniemożliwia konstruowanie nowych
    ' egzemplarzy w bieżącym stanie). Jeśli utworzenie żądanego obiektu nie
    ' jest możliwe, funkcja zwraca wartość Nothing.
    '
    Public Function GetObject() As objectType _
        Implements IObjectPool(Of objectType).GetObject
        SyncLock SyncRoot
            Dim o As objectType = removeObject()
            If o IsNot Nothing Then Return o
            If OkToCreate() Then Return createObject()
            Return Nothing
        End SyncLock
    End Function 'GetObject

    '
    ' Zwraca obiekt składowany w puli. Jeśli w danej chwili pula nie zawiera
    ' żadnych obiektów, funkcja tworzy nowy obiekt (chyba że stosowana w tej
    ' puli polityka tworzenia obiektów uniemożliwia konstruowanie nowych
    ' egzemplarzy w bieżącym stanie). Jeśli utworzenie żądanego obiektu nie
    ' jest możliwe, funkcja czeka, aż któryś z wcześniej przydzielonych obiektów.
    ' zostanie zwrócony i będzie dostępny do ponownego użycia.
    '
    Public Function WaitForObject() As objectType _
        Implements IObjectPool(Of objectType).WaitForObject
        SyncLock SyncRoot
            Dim o As objectType = removeObject()
            If o IsNot Nothing Then Return o
            If OkToCreate() Then Return createObject()
            Do
                ' Czeka na sygnał o zwróceniu obiektu do puli
                ' i możliwości jego ponownego wykorzystania.
                Monitor.Wait(SyncRoot)
                o = removeObject()
            Loop While o Is Nothing
            Return o
        End SyncLock
    End Function 'WaitForObject

    ' Zwraca obiekt do puli.
    Public MustOverride Sub Release(ByVal o As objectType) _
                            Implements IObjectPool(Of objectType).Release
End Class

'
' Klasa SizedObjectPool ogranicza liczbę obiektów składowanych w puli.
'
Public Class SizedObjectPool(Of objectType)
    Inherits AbstractObjectPool(Of objectType)

    ' Ponieważ istnieje nieprzekraczalny limit obiektów oczekujących
    ' w egzemplarzu tej klasy na ponowne wykorzystanie, klasa może używać
    ' do ich składowania prostej tablicy. Klasa dodatkowo definiuje
    ' zmienną egzemplarza nazwaną count, która reprezentuje liczbę obiektów
    ' aktualnie oczekujących na ponowne wykorzystanie.
    Private myCount As Integer

    ' Tablica zawiera obiekty oczekujące na ponowne wykorzystanie.
    ' Tablica jest zarządzana zgodnie z regułami stosowanymi dla stosów.
    Private pool() As objectType

    ' Wewnętrzne operacje na tym obiekcie są synchronizowane. Dokładne
    ' wyjaśnienie zasad funkcjonowania  obiektów blokujących znajdziesz w
    ' podrozdziale poświęconym wzorcowi projektowemu Internal Lock Object.
    Private lockObject As New Object()

    ' Blokowanie obiektu ma na celu zagwarantowanie, że struktura danych
    ' puli będzie jednocześnie przetwarzana tylko przez jeden wątek.
    Protected Overrides ReadOnly Property SyncRoot() As Object
        Get
            Return lockObject
        End Get
    End Property

    ' Zwraca wartość true, jeśli dana pula obiektów może tworzyć
    ' obiekty innej klasy, które będą przez tę pulę zarządzane.
    ' Dla tej puli nie zdefiniowano żadnej strategii ograniczania liczby
    ' jednocześnie składowanych obiektów wielokrotnego użytku, zatem niniejsza
    ' metoda zawsze będzie zwracała wartość true. Gdyby istniała taka strategia,
    ' wynik zwracany przez tę metodę byłby uzależniony od tego, czy liczba
    ' składowanych obiektów jest mniejsza od zdefiniowanego maksimum.
    Protected Overrides Function OkToCreate() As Boolean
        Return True
    End Function

    ' c - Obiektobiekt, do którego dana pula będzie delegowała zadania związane
    '     z tworzeniem obiektów, które będą następnie składowane w tej puli.
    ' m - Maksymalna maksymalna liczba nieużywanych obiektów, które mogą być jednocześnie
    '     składowane w tej puli.
    '
    Public Sub New(ByVal c As ICreation(Of objectType), ByVal m As Integer)
        MyBase.New(c)
        myCount = 0
        pool = New objectType(m) {}
    End Sub

    '
    ' Liczba obiektów aktualnie oczekujących w tej puli na
    ' ponowne wykorzystanie.
    '
    Public Overrides ReadOnly Property Count() As Integer
        Get
            Return myCount
        End Get
    End Property

    ' Maksymalna liczba obiektów, które mogą jednocześnie oczekiwać
    ' w tej puli na ponowne wykorzystanie.
    Public Property Capacity() As Integer
        Get
            Return pool.Length
        End Get
        Set(ByVal Value As Integer)
            If Value <= 0 Then
                Throw New ArgumentException( _
                  "Pojemność puli musi być większa od zera:" & Value)
            End If
            SyncLock SyncRoot
                ReDim Preserve pool(Value)
            End SyncLock
        End Set
    End Property

    ' Usuwa i zwraca na wyjściu obiekt składowany w tablicy pełniącej funkcję puli.
    ' Jeśli dana pula jest pusta, funkcja zwraca wartość Nothing.
    Protected Overrides Function removeObject() As objectType
        myCount -= 1
        If Count >= 0 Then
            Return pool(Count)
        End If
        Return Nothing
    End Function 'removeObject

    '
    ' Zwraca obiekt do puli celem umożliwienia jego ponownego wykorzystania.
    '
    ' o - Obiekt obiekt dostępny do ponownego wykorzystania.
    Public Overrides Sub Release(ByVal o As objectType)
        ' Procedura nie obsługuje wartości null.
        If o Is Nothing Then
            Throw New NullReferenceException()
        End If
        SyncLock SyncRoot
            If Count < Capacity Then
                pool(Count) = o
                myCount += 1
                ' Informuje oczekujący wątek, że do puli trafił
                ' obiekt gotowy do ponownego wykorzystania.
                Monitor.Pulse(SyncRoot)
            End If
        End SyncLock
    End Sub 'Release

End Class 'SizedObjectPool

' Klasy puli obiektów delegują zadanie tworzenia nowych obiektów do
' egzemplarzy tego interfejsu.
Public Interface ICreation(Of objectType)
    ' Zwraca nowo utworzony obiekt.
    Function Create() As objectType
End Interface 'ICreation
